import sys
from transformers import AutoModelForCausalLM, AutoTokenizer
import torch
from PIL import Image
import io, base64, ast, json
import cv2
import numpy as np
sys.path.append('./')
from gui_speaker.models.GUI_R1 import GUI_R1_Agent
from utils.result_preprocess import Agent_CPM_RES_PREPROCESS

class AgentCPM_GUI_Agent(GUI_R1_Agent):
    def __init__(self, device, policy_lm, accelerator):
        self.model = None
        self.policy_lm = policy_lm
        self.res_pre_process = self._res_pre_process()
        self.tokenizer = AutoTokenizer.from_pretrained(self.policy_lm, trust_remote_code=True)
        self.tokenizer.truncation_side = 'left'
        self.device = device
        self.tokenizer.pad_token = self.tokenizer.eos_token
        self.tokenizer.pad_token_id = self.tokenizer.eos_token_id
        self.accelerator = accelerator
        self.generation_config = {
            "temperature": 0.1,            
            "max_tokens": 2048
        }

    def _load_model(self):
        print("AgentCPM load model")
        self.model = AutoModelForCausalLM.from_pretrained(
            self.policy_lm, 
            torch_dtype=torch.bfloat16, 
            device_map="auto", 
            trust_remote_code=True,
            attn_implementation="flash_attention_2",
        ).eval()
        self.model.config.pad_token_id = self.model.config.eos_token_id
        return self.model

    def _res_pre_process(self):
        return Agent_CPM_RES_PREPROCESS()
    
    def encode_image(self, image: Image.Image) -> str:
        """Convert PIL Image to base64-encoded string."""
        with io.BytesIO() as in_mem_file:
            image.save(in_mem_file, format="JPEG")
            in_mem_file.seek(0)
            return base64.b64encode(in_mem_file.read()).decode("utf-8")
    
    def get_action(self, obs, args):
        messages = obs['messages']
        SYSTEM_PROMPT = obs['system_prompt']
        image_path = obs['images'][0]
        image = Image.open(image_path).convert('RGB')
        if args.probing_method == 'visual_mask':
            image = self.visual_mask(image, obs, args)
            output = self._get_action(image, messages, SYSTEM_PROMPT)
            return output
        elif args.probing_method == 'zoom':
            image, label = self.zoom_in(image, obs)
            output = self._get_action(image, messages, SYSTEM_PROMPT)
            label["label"] = json.dumps({
                "thought": "",
                "POINT": [label["label"][0], label["label"][1]]
            })
            return output, label
        elif args.probing_method == 'visual_edit':
            image = self.visual_mask(image, obs, args)
            output = self._get_action(image, messages, SYSTEM_PROMPT)
            return output
        elif args.probing_method == 'structure_mask' and args.dataset_type == 'action_shortcuts':
            image = self.structure_mask(image)
            output = self._get_action(image, messages, SYSTEM_PROMPT)
            return output
        else:
            output = self._get_action(image, messages, SYSTEM_PROMPT)
            return output
    
    def _get_action(self, image, messages, SYSTEM_PROMPT):
        image = self.__resize__(image)
        messages[0]['content'].append(
            image
        )
        output_text = self.model.chat(
            image=None, msgs=messages, system_prompt=SYSTEM_PROMPT, tokenizer=self.tokenizer, temperature=0.1,top_p=0.3,n=1, max_new_tokens = 512,
        )

        return output_text
    
    def __resize__(self, origin_img):
            resolution = origin_img.size
            w,h = resolution
            max_line_res = 1120
            if max_line_res is not None:
                max_line = max_line_res
                if h > max_line:
                    w = int(w * max_line / h)
                    h = max_line
                if w > max_line:
                    h = int(h * max_line / w)
                    w = max_line
            img = origin_img.resize((w,h),resample=Image.Resampling.LANCZOS)
            return img
    
    def visual_mask(self, image_input, obs, args):
        from PIL import ImageDraw
        draw = ImageDraw.Draw(image_input)
        image_width, image_height = image_input.size[0], image_input.size[1]
        if obs.get('dataset_name') == 'AndroidControl':
            accessibility_trees_file_path = obs['accessibility_trees']
            bbox_data = []
            with open(accessibility_trees_file_path, "r", encoding="utf-8") as file:
                for line in file:
                  try:
                      obj = json.loads(line)
                      bbox = obj.get("bbox_pixels", None)
                      class_name = obj.get("class_name", None)
                      if bbox and (class_name == 'android.widget.ImageButton' or class_name == 'android.widget.TextView' or class_name == 'android.widget.ImageView') and obj.get("is_clickable"):
                          x_min, y_min, x_max, y_max = bbox["x_min"], bbox["y_min"], bbox["x_max"], bbox["y_max"]
                          if (
                              0 <= x_min < x_max <= image_width and
                              0 <= y_min < y_max <= image_height
                          ):
                              bbox_data.append([x_min, y_min, x_max-x_min, y_max-y_min])
                  except Exception:
                      continue
            gt = self.res_pre_process.extract_action(obs.get('label'))
            gt = self.res_pre_process.extract_coordinates(gt)
        elif obs.get('dataset_name') == 'AITZ':
            accessibility_trees_file_path = obs['accessibility_trees']
            bbox_data = []
            with open(accessibility_trees_file_path, "r", encoding="utf-8") as file:
                accessibility_trees_file_data = json.load(file)
            for idx, acc_data in enumerate(accessibility_trees_file_data):
                if acc_data['image_path'] in obs.get('images')[0]:
                    bbox = ast.literal_eval(accessibility_trees_file_data[idx]['ui_positions'])
            bbox_data = [[y, x, h, w] for (x, y, w, h) in bbox]
          
        else:
            bbox_data = obs.get('bbox')
            bbox_data = [bbox_data[0]/1000*image_width, bbox_data[1]/1000*image_height, bbox_data[2]/1000*image_width, bbox_data[3]/1000*image_height]
            bbox_data = [[bbox_data[0], bbox_data[1], bbox_data[2]-bbox_data[0], bbox_data[3]-bbox_data[1]]]
        gt = self.res_pre_process.extract_action(obs['label'])
        gt = self.res_pre_process.extract_coordinates(gt)
        _, bbox_list, point = self.remove_containing_bboxes(bbox_list=bbox_data, gt=gt, image_size=[image_width, image_height]) 
        if args.probing_method == 'visual_mask':
            if len(bbox_list) > 0:
                for bbox in bbox_list:
                    x, y, w, h = bbox
                    draw.rectangle([x, y, x+w, y+h], fill="black")
            else:
                r = args.mask_object_ratio
                draw.rectangle([point[0]-r, point[1]-r, point[0]+r, point[1]+r], fill="black")
        else:
            image_cv = np.array(image_input)
            image_input = cv2.cvtColor(image_cv, cv2.COLOR_RGB2BGR)
            if len(bbox_list) > 0:
                for bbox in bbox_list:
                    mask = np.zeros(image_input.shape[:2], dtype=np.uint8)
                    mask[int(bbox[1]):int(bbox[3]), int(bbox[0]):int(bbox[2])] = 255
                    image_input = cv2.inpaint(image_input, mask, 3, cv2.INPAINT_TELEA)
                # image_input = Image.fromarray(image_input)
            r = args.mask_object_ratio
            mask = np.zeros(image_input.shape[:2], dtype=np.uint8)
            x, y = point
            x_min = int(x - r)
            y_min = int(y - r)
            x_max = int(x + r)
            y_max = int(y + r)
            mask[y_min:y_max, x_min:x_max] = 255

            image_input = cv2.inpaint(image_input, mask, 3, cv2.INPAINT_TELEA)
            image_input = Image.fromarray(image_input)
        return image_input

    def remove_containing_bboxes(self, bbox_list, gt, image_size):
        click_x, click_y = gt[0]/1000*image_size[0], gt[1]/1000*image_size[1]
        out_bbox_list = []
        in_bbox_list = []
        if len(bbox_list) > 0:
            for bbox in bbox_list:
                x, y, w, h = bbox
                if not (x <= click_x <= x+w and y <= click_y <= y+h):
                    out_bbox_list.append(bbox)
                else:
                    in_bbox_list.append(bbox)
        return out_bbox_list, in_bbox_list, (click_x, click_y)
    

    def zoom_in(self, pil_image, obs):
        from PIL import Image

        try:
            content = obs['label']
        except (IndexError, KeyError, TypeError):
            raise ValueError("Invalid message format in obs")

        ground_truth = self.res_pre_process.extract_action(content)
        task = self.res_pre_process.get_action_type(ground_truth)
        bbox = obs.get("bbox")  # [x_min, y_min, x_max, y_max]

        w, h = pil_image.size

        if task == 1:
            click_x, click_y = self.res_pre_process.extract_coordinates(ground_truth)
            click_x = click_x / 1000 * w
            click_y = click_y / 1000 * h

            mid_x, mid_y = w // 2, h // 2
            if click_x < mid_x and click_y < mid_y:
                region = (0, 0, mid_x, mid_y)
            elif click_x >= mid_x and click_y < mid_y:
                region = (mid_x, 0, w, mid_y)
            elif click_x < mid_x and click_y >= mid_y:
                region = (0, mid_y, mid_x, h)
            else:
                region = (mid_x, mid_y, w, h)

            cropped = pil_image.crop(region)
            zoomed_image = cropped.resize((w, h), Image.LANCZOS)

            def transform_coord(x, y, region, w, h):
                rel_x, rel_y = x - region[0], y - region[1]
                scale_x = w / (region[2] - region[0])
                scale_y = h / (region[3] - region[1])
                new_x = int(rel_x * scale_x)
                new_y = int(rel_y * scale_y)
                return new_x, new_y

            new_click_x, new_click_y = transform_coord(click_x, click_y, region, w, h)
            norm_click_x = new_click_x / w * 1000
            norm_click_y = new_click_y / h * 1000
            
            new_bbox = None
            if bbox is not None:
                bbox = [bbox[0]/1000*w, bbox[1]/1000*h, bbox[2]/1000*w, bbox[3]/1000*h]
                x_min, y_min = transform_coord(bbox[0], bbox[1], region, w, h)
                x_max, y_max = transform_coord(bbox[2], bbox[3], region, w, h)
                new_bbox = [x_min/w*1000, y_min/h*1000, x_max/w*1000, y_max/h*1000]

            return zoomed_image, {"label": [norm_click_x, norm_click_y], "bbox": new_bbox}

        return pil_image, None
    
        
    def structure_mask(self, image):
        masked_image = np.zeros_like(image)
        image_input = Image.fromarray(masked_image)
        return image_input

